﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace MicroRWD.QT
{
    // Encapsulates the status byte
    public class Status
    {
        #region Public Properties

        // b0 Internal EEPROM write error
        public bool EEPROM_Error { get; set; }
        // b1 Tag identity code matched to list
        public bool TagOK { get; set; }
        // b2 Tag communication and acknowledgement OK
        public bool RxOK { get; set; }
        // b3 Host serial communication error
        public bool RS232_Error { get; set; }
        // b4 Relay enabled flag
        public bool RelayEnabled { get; set; }
        // b5 HTRC (or Antenna fault) error flag
        public bool HTRC_AntennaFaultError { get; set; }
        // Validity
        public bool Valid { get; set; }

        #endregion


        #region Constructors

        // Construct an empty invalid Status instance
        public Status()
        {
            SetValue(0);
        }

        // Construct an initialised Status instance
        public Status(byte status)
        {
            SetValue(status);
        }

        #endregion


        #region Public Methods

        // Get the byte value from the current state
        public byte GetValue()
        {
            byte result;

            if (Valid)
            {
                // All valid MicroRWD.QT responses have b7 and b6 set
                result = 0xC0;
                // b0 Internal EEPROM write error
                result |= (byte)(EEPROM_Error ? (1 << 0) : 0);
                // b1 Tag identity code matched to list
                result |= (byte)(TagOK ? (1 << 1) : 0);
                // b2 Tag communication and acknowledgement OK
                result += (byte)(RxOK ? (1 << 2) : 0);
                // b3 Host serial communication error
                result |= (byte)(RS232_Error ? (1 << 3) : 0);
                // b4 Relay enabled flag
                result |= (byte)(RelayEnabled ? (1 << 4) : 0);
                // b5 HTRC (or Antenna fault) error flag
                result |= (byte)(HTRC_AntennaFaultError ? (1 << 5) : 0);
            }
            else
            {
                // All bits zero indicates lack of a response
                result = 0x00;
            }

            // Return result;
            return result;
        }

        // Set the current state based on a byte value
        public void SetValue(byte _status)
        {
            // All valid MicroRWD responses have b7 and b6 set
            if ((_status & 0xC0) == 0xC0)
            {
                // b0 Internal EEPROM write error
                EEPROM_Error = (_status & (1 << 0)) != 0;
                // b1 Tag identity code matched to list
                TagOK = (_status & (1 << 1)) != 0;
                // b2 Tag communication and acknowledgement OK
                RxOK = (_status & (1 << 2)) != 0;
                // b3 Host serial communication error
                RS232_Error = (_status & (1 << 3)) != 0;
                // b4 Relay enabled flag
                RelayEnabled = (_status & (1 << 4)) != 0;
                // b5 HTRC (or Antenna fault) error flag
                HTRC_AntennaFaultError = (_status & (1 << 5)) != 0;
                // Valid
                Valid = true;
            }
            else
            {
                // b0 Internal EEPROM write error
                EEPROM_Error = false;
                // b1 Tag identity code matched to list
                TagOK = false;
                // b2 Tag communication and acknowledgement OK
                RxOK = false;
                // b3 Host serial communication error
                RS232_Error = false;
                // b4 Relay enabled flag
                RelayEnabled = false;
                // b5 HTRC (or Antenna fault) error flag
                HTRC_AntennaFaultError = false;
                // Invalid
                Valid = false;
            }
        }

        // Express the Status as a bit string
        public override string ToString()
        {
            // Somewhere to hold the result
            string result;

            // Convert to bits
            if (Valid)
            {
                // All valid MicroRWD responses have b7 and b6 set
                result = "1 1 ";
            }
            else
            {
                // All bits zero indicates lack of a response
                result = "0 0 ";
            }

            // b5 HTRC (or Antenna fault) error flag
            result += HTRC_AntennaFaultError ? "1 " : "0 ";
            // b4 Relay enabled flag
            result += RelayEnabled ? "1 " : "0 ";
            // b3 Host serial communication error
            result += RS232_Error ? "1 " : "0 ";
            // b2 Tag communication and acknowledgement OK
            result += RxOK ? "1 " : "0 ";
            // b1 Tag identity code matched to list
            result += TagOK ? "1 " : "0 ";
            // b0 Internal EEPROM write error
            result += EEPROM_Error ? "1" : "0";
            
            // Return result
            return result;
        }

        #endregion
    }
}
